import QtQuick 2.2
import QtGraphicalEffects 1.0
import QtQml 2.2

import wesual.Controls  1.0

Item {
    id : datePicker

    property alias text    : dateText.text
    property date  selectedDate
    property alias popup : calendarPopup
    property alias minimumDate : calendarPopup.minimumDate
    property alias maximumDate : calendarPopup.maximumDate

    signal dateChosen()

    implicitWidth  : 110
    implicitHeight : 26

    QtObject {
        id : p_

        readonly property int   iconWidth      : 25
        readonly property int   iconHeight     : 26
        readonly property int   animationDuration : 0
        property bool           reset : false
    }

    onSelectedDateChanged : {
        if (p_.reset)
            return;

        var today = new Date();

        if (selectedDate.getDate()     === today.getDate()  &&
            selectedDate.getMonth()    === today.getMonth() &&
            selectedDate.getFullYear() === today.getFullYear()) {
            dateText.text = calendarPopup.buttonText;
        } else {
            var locale = Qt.locale();
            dateText.text =
                 selectedDate.toLocaleDateString(locale,
                                                 getAdaptedLocalFormat(locale));
        }
        if (!isNaN(selectedDate.getTime()))
            calendarPopup.selectedDate = selectedDate
        else
            calendarPopup.selectedDate = new Date();


        dateChosen();
    }

    Rectangle {
        id : background

        color : UI.color(UI.PrimaryBase)
        border.color : {
            if (!datePicker.enabled)
                return UI.color(UI.DisabledBorder);

            if (mouseAreaIcon.pressed || mouseAreaText.pressed || popup.open ||
                dateText.activeFocus)
                return UI.color(UI.ControlBorderPress)
            else if (mouseAreaIcon.containsMouse ||
                mouseAreaText.containsMouse)
                return UI.color(UI.ControlBorderHover);
            else
                return UI.color(UI.PrimaryControlBorder);
        }
        border.width : 1
        anchors.fill : parent

        Behavior on border.color {
            ColorAnimation { duration : p_.animationDuration }
        }
    }

    TextField {
        id : dateText

        width  : parent.width - p_.iconWidth
        height : parent.height

        border.width : 0
        backgroundColor : "transparent"

//        validator : RegExpValidator {
//            regExp : /^(0[1-9]|[12][0-9]|3[01])[- /. //](0[1-9]|1[012])[- /. //](19|20)\d\d$/
//        }

        onEditFinished : {
            if (p_.reset)
                return;

            var dateString = dateText.text;
            var locale = Qt.locale();
            var format = getAdaptedLocalFormat(locale);
            var dateFromString = Date.fromLocaleDateString(locale,
                                                           dateString,
                                                           format);

            // if date string is invalid -> display last valid date
            if (isNaN(dateFromString.getTime())) {
                if (dateText.text.toLocaleLowerCase() ===
                    calendarPopup.buttonText.toLowerCase()) {
                    dateText.text = calendarPopup.buttonText;
                    selectedDate = new Date();
                } else {
                    if (selectedDate)
                        dateText.text = selectedDate.toLocaleDateString(locale,
                                                 getAdaptedLocalFormat(locale));
                    else
                        dateText.text = "";
                }
            } else {
                var today = new Date();

                // date is equal to today or less than the minimum date
                if (dateFromString.getDate()     === today.getDate()     &&
                    dateFromString.getMonth()    === today.getMonth()    &&
                    dateFromString.getFullYear() === today.getFullYear() &&
                    +dateFromString < +minimumDate) {
                    dateText.text = calendarPopup.buttonText;
                } else if (+dateFromString < +minimumDate) {
                    selectedDate = minimumDate;
                } else if (+dateFromString > +maximumDate) {
                    selectedDate = maximumDate;

                } else {
                    selectedDate = dateFromString;
                }
            }
        }
    }

    Rectangle {
        id : icon

        width  : p_.iconWidth  - 1
        height : p_.iconHeight - 2

        anchors {
            top   : parent.top
            right : parent.right
            rightMargin : 1
            topMargin   : 1
        }

        color : {
            if (!datePicker.enabled)
                return UI.color(UI.PrimaryBase);

            if (mouseAreaIcon.pressed || calendarPopup.popupVisible)
                return UI.color(UI.PrimaryPress);
            else if (mouseAreaIcon.containsMouse)
                return UI.color(UI.PrimaryHover);
            else
                return UI.color(UI.PrimaryBase);
        }

        Behavior on color {
            ColorAnimation { duration: 0 }
        }

        ColorizedImage {
            id : calIcon
            anchors.top   : parent.top
            anchors.right : parent.right
            anchors.rightMargin : -1
            anchors.topMargin   : -1
            sourceSize {
                width  : p_.iconWidth
                height : p_.iconHeight
            }
            smooth : true
            source : "qrc:/controls/icons/calendar.png"
            color   : {
                if (!datePicker.enabled)
                    return "#DFDFDF";

                if (mouseAreaIcon.pressed || calendarPopup.popupVisible)
                    return UI.color(UI.PrimaryBase);
                else
                    return UI.color(UI.CalendarIcon);
            }
        }
    }

    PopupAnchorItem {
        id : calendarPopup

        property date maximumDate  : new Date(2999,11,31)
        property date minimumDate  : new Date(1901,0,1)
        property string buttonText : qsTrId("8298533db336ff8c")
        property string pickerText : dateText.text
        property date selectedDate : new Date()

        onPopupVisibleChanged : {
            if (popupVisible && popupItem) {
                popupItem.selectedDate = selectedDate;
            }
        }

        anchors.centerIn: parent

        popup : Calendar {
            id : cal

            maximumDate  : calendarPopup.maximumDate
            minimumDate  : calendarPopup.minimumDate
            buttonText   : calendarPopup.buttonText
            selectedDate : calendarPopup.selectedDate
            pickerText   : calendarPopup.pickerText

            readonly property bool northSouthFlip  :
                        calendarPopup.distanceBottom < cal.implicitHeight
            readonly property int  preferredHeight : cal.implicitHeight

            height : Math.min(
                         preferredHeight,
                         (northSouthFlip ?
                         calendarPopup.distanceTop :
                         calendarPopup.distanceBottom))

            anchors {
                top : northSouthFlip ? undefined : parent.bottom
                bottom : northSouthFlip ? parent.top : undefined
                left : parent.left
                topMargin : calendarPopup.parent.height / 2 - 1
                bottomMargin : calendarPopup.parent.height / 2 + 6
                leftMargin : -calendarPopup.parent.width / 2
            }

            opacity : 0

            states : [
                State {
                    name : "visible"
                    when : calendarPopup.popupVisible

                    PropertyChanges {
                        target : cal
                        opacity : 1
                    }
                    PropertyChanges {
                        target : cal
                        focus  : true
                    }
                }
            ]

            transitions : Transition {
                NumberAnimation {
                    property : "opacity"
                    duration : 200
                }
            }

            onClicked : {
                var today = new Date();

                if (+selectedDate === +today) {
                    dateText.text = cal.buttonText;
                    datePicker.selectedDate = today;
                }
                if (+selectedDate === +datePicker.selectedDate)
                    dateChosen();
                else
                    datePicker.selectedDate = selectedDate;

                calendarPopup.hide();
            }
        }
    }

    MouseArea {
        id : mouseAreaIcon

        anchors.fill : icon
        hoverEnabled : enabled
        enabled      : !calendarPopup.popupItem ||
                       calendarPopup.popupItem.opacity === 0
        onPressed    : calendarPopup.show()
        cursorShape  : Qt.PointingHandCursor
    }

    MouseArea {
        id : mouseAreaText

        anchors.fill : dateText
        hoverEnabled : datePicker.enabled
        acceptedButtons: Qt.NoButton
        cursorShape : Qt.IBeamCursor
    }

    function resetDatePicker() {
        p_.reset = true;
        dateText.text = "";
        selectedDate = new Date("invalid date");
        minimumDate  = new Date(1901,0,1);
        maximumDate  = new Date(2999,11,31);
        calendarPopup.selectedDate = new Date();
        p_.reset = false;
    }

    function getAdaptedLocalFormat(locale) {
        var format = locale.dateFormat(Locale.ShortFormat);

        var leadingZeros = format.indexOf("dd");
        if (leadingZeros !== -1)
            format = format.replace("dd", "d");

        leadingZeros = format.indexOf("MM");
        if (leadingZeros !== -1)
            format = format.replace("MM", "M");

        leadingZeros = format.indexOf("yyyy");
        if (leadingZeros === -1)
            format = format.replace("yy", "yyyy");

        return format;
    }
}


